import axios from "axios";
import React, { useEffect, useState } from "react";
import ReactDOM from "react-dom";

import "react-progress-2/main.css";
import Progress from "react-progress-2";
import toast from "react-hot-toast";
import ProductList from "../components/ProductList";
import { Col, Container, Row } from "react-bootstrap";
import MainCategory from "../components/components/MainCategory";
import FilterBrand from "../components/components/FilterBrand";
import FilterWeight from "../components/components/FilterWeight";
import AvailableShop from "../components/components/AvailableShop";
import MobileFilter from "../components/components/MobileFilter";
import FilterList from "../components/FilterList";

const Search = (props) => {
    const searchId = props.slug;
    const [loading, setLoading] = useState(false);
    const [products, setProducts] = useState([]);
    const [sizes, setSizes] = useState([]);
    const [categories, setCategories] = useState([]);
    const [brands, setBrands] = useState([]);

    const [totalProducts, setTotalProducts] = useState(0);
    const [perPages, setPerPages] = useState(0);
    const [currentPage, setCurrentPage] = useState(0);

    const [sort, setSort] = useState("asc");

    const loadData = async () => {
        setLoading(true);

        await axios
            .get(`/api/searches?query=${searchId}`)
            .then((response) => {
                if (response.data.result === "success") {
                    setBrands(response.data.brands);
                    setProducts(response.data.products);
                    setCategories(response.data.categories);
                    setSizes(response.data.sizes);
                    setTotalProducts(response.data.totalProducts);
                    setPerPages(response.data.perPages);
                    setCurrentPage(response.data.currentPage);
                }
            })
            .catch((error) => {
                console.log(error.message);
            });

        setLoading(false);
    };

    useEffect(() => {
        loadData();
    }, []);

    const [selectedCuisines, setSelectedCuisines] = useState([]);
    const [selectedCategories, setSelectedCategories] = useState([]);

    const [selectedSizes, setSelectedSizes] = useState([]);
    const [checkedSizes, setCheckedSizes] = useState([]);

    const [selectedBrands, setSelectedBrands] = useState([]);
    const [checkedBrands, setCheckedBrands] = useState([]);

    const getData = async (pageNumber = 1) => {
        window.scrollTo(0, 0);
        Progress.show();
        setLoading(true);

        try {
            const filters = [];
            const sizeFilters = [];
            if (selectedCuisines.length) {
                filters.push(`categories=${selectedCuisines.join(",")}`);
            }
            if (selectedBrands.length) {
                filters.push(`brands=${selectedBrands.join(",")}`);
            }
            if (selectedSizes.length) {
                sizeFilters.push(`sizes=${selectedSizes.join(",")}`);
            }
            const filterQueryString =
                filters.length > 0 ? `&${filters.join("&")}` : "";
            const filterQuerySizes =
                filters.length > 0 ? `&${filters.join("&")}` : "";
            const filterQueryBrands =
                filters.length > 0 ? `&${filters.join("&")}` : "";
            const url = `/api/searches/filters?query=${searchId}&${filterQueryString}&sizes=${filterQuerySizes}&brands=${filterQueryBrands}&page=${pageNumber}&sort=${sort}`;

            const response = await axios.get(url);

            setProducts(response.data.products);
            setTotalProducts(response.data.totalProducts);
            setPerPages(response.data.itemsCountPerPage);
            setCurrentPage(response.data.currentPage);

            Progress.hide();
            setLoading(false);
        } catch (error) {
            toast.error(error.message);
            Progress.hide();
            setLoading(false);
        }
    };

    const handleChangeCategory = (id, type) => {
        const updatedCategories = categories.map((item) => {
            if (type === "category") {
                return item.id === id
                    ? { ...item, checked: !item.checked }
                    : item;
            } else if (type === "subCategory") {
                const updatedSubCategory = item.subCategory.map(
                    (subCategory) => {
                        return subCategory.id === id
                            ? { ...subCategory, checked: !subCategory.checked }
                            : subCategory;
                    }
                );
                return { ...item, subCategory: updatedSubCategory };
            } else if (type === "childCategory") {
                const updatedSubCategory = item.subCategory.map(
                    (subCategory) => {
                        const updatedChildCategory =
                            subCategory.childCategory.map((childCategory) => {
                                return childCategory.id === id
                                    ? {
                                          ...childCategory,
                                          checked: !childCategory.checked,
                                      }
                                    : childCategory;
                            });
                        return {
                            ...subCategory,
                            childCategory: updatedChildCategory,
                        };
                    }
                );
                return { ...item, subCategory: updatedSubCategory };
            }
            return item;
        });

        setCategories(updatedCategories);
    };

    const handleChangeSize = (id, type) => {
        const updateSizes = sizes.map((item) => {
            if (type === "size") {
                return item.id === id
                    ? { ...item, checked: !item.checked }
                    : item;
            }
            return item;
        });

        setSizes(updateSizes);
    };

    const handleChangeBrand = (id, type) => {
        const updateBrands = brands.map((item) => {
            if (type === "brand") {
                return item.id === id
                    ? { ...item, checked: !item.checked }
                    : item;
            }
            return item;
        });

        setBrands(updateBrands);
    };

    const changePrice = async () => {
        const filters = [];
        const sizeFilters = [];
        const brandFilters = [];

        const cuisinesChecked = categories
            .filter((item) => item.checked)
            .map((item) => ({
                id: item.id,
                title: item.title,
                type: "category",
            }));
        const subCategoriesChecked = categories
            .flatMap((item) =>
                item.subCategory.filter((subItem) => subItem.checked)
            )
            .map((subItem) => ({
                id: subItem.id,
                title: subItem.title,
                type: "subCategory",
            }));
        const childCategoriesChecked = categories
            .flatMap((item) =>
                item.subCategory.flatMap((subItem) =>
                    subItem.childCategory.filter(
                        (childItem) => childItem.checked
                    )
                )
            )
            .map((childItem) => ({
                id: childItem.id,
                title: childItem.title,
                type: "childCategory",
            }));
        const sizesChecked = sizes
            .filter((item) => item.checked)
            .map((item) => ({
                id: item.id,
                title: item.title,
                subtitle: item.subtitle,
                type: "size",
            }));
        const brandChecked = brands
            .filter((item) => item.checked)
            .map((item) => ({
                id: item.id,
                title: item.title,
                type: "brand",
            }));

        const allCheckedIds = [
            ...cuisinesChecked,
            ...subCategoriesChecked,
            ...childCategoriesChecked,
        ];

        const selectedCuisinesIds = allCheckedIds.map((item) => item.id);
        setSelectedCuisines(selectedCuisinesIds);
        setSelectedCategories(allCheckedIds);

        const selectedBrandIds = brandChecked.map((item) => item.id);
        setSelectedBrands(brandChecked);
        setCheckedBrands(brandChecked);

        const selectedSizeIds = sizesChecked.map((item) => item.subtitle);
        setSelectedSizes(sizesChecked);
        setCheckedSizes(sizesChecked);

        if (allCheckedIds.length) {
            filters.push(`categories=${selectedCuisinesIds.join(",")}`);
        }

        if (selectedBrandIds.length) {
            brandFilters.push(`brands=${selectedBrandIds.join(",")}`);
        }

        if (selectedSizeIds.length) {
            sizeFilters.push(`sizes=${selectedSizeIds.join(",")}`);
        }

        const apiUrl = `/api/searches/filters?query=${searchId}&${
            filters.length > 0 ? filters.join("&") : ""
        }${brandFilters.length > 0 ? `&` + brandFilters.join("&") : ""}${
            sizeFilters.length > 0 ? `&` + sizeFilters.join("&") : ""
        }${sort ? `&sort=${sort}` : ""}`;
        console.log(apiUrl);

        await axios
            .get(apiUrl)
            .then((response) => {
                setProducts(response.data.products);
                setTotalProducts(response.data.totalProducts);
                setPerPages(response.data.itemsCountPerPage);
                setCurrentPage(response.data.currentPage);
            })
            .catch((error) => {
                toast.error(error.message);
            });
    };

    useEffect(() => {
        changePrice();
    }, [totalProducts, categories, sizes, sort, brands]);

    const [show, setShow] = useState(false);

    const handleClose = () => setShow(false);
    const handleShow = () => setShow(true);

    return (
        <>
            <Progress.Component
                style={{ background: "#99999978", height: "5px" }}
                thumbStyle={{
                    background: "#f29f05",
                    height: "5px",
                }}
            />
            <section className="product section-gap">
                <Container>
                    <Row className="g-4">
                        <FilterList
                            handleShow={handleShow}
                            sort={sort}
                            setSort={setSort}
                        />
                        <Col
                            lg={3}
                            className="hidden desktop__filter"
                            id="filter-sidebar"
                        >
                            <div className="service-sidebar">
                                <FilterBrand
                                    loading={loading}
                                    brands={brands}
                                    handleChange={handleChangeBrand}
                                />
                                <MainCategory
                                    loading={loading}
                                    categories={categories}
                                    handleChange={handleChangeCategory}
                                />
                            </div>
                        </Col>

                        <Col lg={12} id="products-col">
                            <div className="th-sort-bar">
                                <div className="store__container">
                                    <div className="col-md desktop__filter">
                                        {totalProducts} products found{" "}
                                        {searchId && <>on {searchId}</>}
                                    </div>
                                </div>
                            </div>
                            <div>
                                <div className="mobile__filter mb-3">
                                    {totalProducts} products found{" "}
                                    {searchId && <>on {searchId}</>}
                                </div>
                                <ul className="category__list">
                                    {checkedBrands.length > 0 &&
                                        checkedBrands.map((item, index) => (
                                            <li key={index}>
                                                <div className="category__item">
                                                    {item.title}
                                                    <label
                                                        onClick={() =>
                                                            handleChangeBrand(
                                                                item.id,
                                                                item.type
                                                            )
                                                        }
                                                        className="filter-summary-removeFilter"
                                                    >
                                                        <span className="filter-summary-removeIcon sprites-remove">
                                                            <i className="fal fa-times"></i>
                                                        </span>
                                                    </label>
                                                </div>
                                            </li>
                                        ))}
                                    {selectedCategories.length > 0 &&
                                        selectedCategories.map(
                                            (item, index) => (
                                                <li key={index}>
                                                    <div className="category__item">
                                                        {item.title}
                                                        <label
                                                            onClick={() =>
                                                                handleChange(
                                                                    item.id,
                                                                    item.type
                                                                )
                                                            }
                                                            className="filter-summary-removeFilter"
                                                        >
                                                            <span className="filter-summary-removeIcon sprites-remove">
                                                                <i className="fal fa-times"></i>
                                                            </span>
                                                        </label>
                                                    </div>
                                                </li>
                                            )
                                        )}
                                </ul>
                            </div>

                            <ProductList
                                loading={loading}
                                products={products}
                                totalProducts={totalProducts}
                                currentPage={currentPage}
                                perPages={perPages}
                                getData={getData}
                            />
                        </Col>
                    </Row>

                    <MobileFilter
                        show={show}
                        loading={loading}
                        handleClose={handleClose}
                        categories={categories}
                        handleChange={handleChangeCategory}
                        sizes={sizes}
                        brands={brands}
                        handleChangeBrand={handleChangeBrand}
                        handleChangeSize={handleChangeSize}
                    />
                </Container>
            </section>
        </>
    );
};

export default Search;

if (document.getElementById("searchFilterContainer")) {
    ReactDOM.render(
        <Search {...searchFilterContainer.dataset} />,
        document.getElementById("searchFilterContainer")
    );
}
